#include "stdafx.h"
#include "common.h"
#include "tree.h"
#include <stdio.h>
#include "opcodes.h"
#include "scriptHeaders.h"
#include "error.h"
#include "asmutils.h"
#include "constanttable.h"
#include "assembler.h"


const int START_CODE_SIZE = 100;	//pocatecni velikost kodoveho segmentu

//****************************************************************************************************************
//Konstruktor
//****************************************************************************************************************
CLabelsList::~CLabelsList()
{
	m_List.clear();
}
//****************************************************************************************************************
//Fce zjisti jestli je navesti obsazeno v seznamu
//****************************************************************************************************************
int		CLabelsList::Contains(char* uniquename)
{
	for (unsigned int i = 0; i < (unsigned int)m_List.size(); i++)
		if (*((LABEL_ADRESSES*)(m_List[i])) == uniquename)				//jen pruchod polem v cyklu a porovnavani se vsemi prvky
			return i;

	return -1;
}
//****************************************************************************************************************
//Pridani navesti do pole
//****************************************************************************************************************
HRESULT	CLabelsList::AddLabel(char* uniquename, int iOffset)
{
	//kontrola jestli uz neni v poli
	int index = Contains(uniquename);
	if (index >= 0)
		return S_FALSE;

	//v poli neni, pridej navesti do pole
	LABEL_ADRESSES* pLabel = new LABEL_ADRESSES;
	pLabel->unique_labelname = uniquename;
	pLabel->label_offset = iOffset;

	m_List.push_back(pLabel);

	return S_OK;
}
//****************************************************************************************************************
//Fce pro vraceni ukazatele na navesti
//****************************************************************************************************************
LABEL_ADRESSES*		CLabelsList::GetLabel(char* uniquename)
{
	int index = Contains(uniquename);		//najdeme jeho pozici pomoci Contains
	if (index >= 0)
		return m_List[index];				//vratime ukazatel pokud byla vracena pozice platna

	return NULL;							//vratime NULL pokud byla vracena pozice neplatna
}
//****************************************************************************************************************
//zikani adresy navesti
//****************************************************************************************************************
int		CLabelsList::GetLabelAdress(char* uniquename)
{
	LABEL_ADRESSES* pLabel = GetLabel(uniquename);	//jen volame predchozi funkci

	if (pLabel)								//pokud navesti v seznamu je, bude pLabel platny ukazatel
		return pLabel->label_offset;		//staci jen vratit pozadovanou polozku

	return -1;								//ukazatel byl neplatny, vratime neplatnou adresu
}
//****************************************************************************************************************
//zjisti velikost seznamu
//****************************************************************************************************************
unsigned int CLabelsList::Size()
{
	return (unsigned int)m_List.size();
}
//****************************************************************************************************************
//operator pro vyber polozky
//****************************************************************************************************************
LABEL_ADRESSES*	CLabelsList::operator [] (unsigned int index)
{
	if (index > (unsigned int)m_List.size())
		return NULL;

	return m_List[index];
}

//****************************************************************************************************************
//****************************************************************************************************************
//****************************************Trida CAssembler********************************************************
//****************************************************************************************************************
//****************************************************************************************************************

//****************************************************************************************************************
//Konstruktor
//****************************************************************************************************************
CAssembler::CAssembler()
{
	m_pByteCode = NULL;				//nulovani promennych
	m_iCurrMult = 0;
	m_iCurrPositionInCode = 0;

	m_binFile = m_inputFile = NULL;
	m_CurrFuncName = NULL;
}

CAssembler::~CAssembler()
{
}

//****************************************************************************************************************
//fce ktera zkontroluje jestli se do pole vejde este nBytu a pokud ne tak ho zvetsi
//****************************************************************************************************************
void CAssembler::CheckCodeSize(int nBytes)
{
	unsigned char* temp;

	//kontrolujeme zda je bytovy kodovy segment dost velky aby se do nej ulozilo jeste nBytes bytu
	//Abychom pole nezvetsovali na dvojnasobek puvodni velikosti, mame ulozen soucasny multiplikator velikosti
	//a pole tak vzdy v pripade potreby zvetsime o definovanou pocatecni velikost START_CODE_SIZE
	if ((m_iCurrPositionInCode + nBytes) >= (m_iCurrMult * START_CODE_SIZE))	//kontrola mezi
	{
		m_iCurrMult++;
		temp = new unsigned char[m_iCurrMult * START_CODE_SIZE];			//zvetseni pole o START_CODE_SIZE
		if (m_pByteCode)
			memcpy((char*)temp,(char*)m_pByteCode,(m_iCurrMult - 1)  * START_CODE_SIZE);						//prekopirovani puvodniho pole
		m_pByteCode = temp;												//nastaveni noveho pole na stare
	}
}

//****************************************************************************************************************
//fce vrati true pokud symbol je navesti
//****************************************************************************************************************
bool		IsLabel(char* token)
{
	if (token[strlen(token) - 1] == ':')
		return true;
	
	return false;
}
//****************************************************************************************************************
//fce pro pridani bin kodu instrukce do pole, kodoveho segmentu
//****************************************************************************************************************
void CAssembler::AddCode(unsigned char code)
{
	CheckCodeSize(1);							//zkontroluj velikost pole, zda se tam este vejdem

	m_pByteCode[m_iCurrPositionInCode] = code;	//uloz kod instrukce do segmentu
	m_iCurrPositionInCode++;						//zvys pozici v kodu, kde jsme
}
//****************************************************************************************************************
//Fce pro pridani 16bitoveho indexu do kodoveho segmentu
//****************************************************************************************************************
void CAssembler::AddIndex16(unsigned short index)
{
	CheckCodeSize(sizeof(unsigned short));

	*((unsigned short*)(&m_pByteCode[m_iCurrPositionInCode])) = index;

	m_iCurrPositionInCode += sizeof(unsigned short);
}
//****************************************************************************************************************
//Fce pro pridani 32bitoveho indexu do kodoveho segmentu
//****************************************************************************************************************
void CAssembler::AddIndex32(unsigned int index)
{
	CheckCodeSize(sizeof(unsigned int));

	*((unsigned int*)(&m_pByteCode[m_iCurrPositionInCode])) = index;

	m_iCurrPositionInCode += sizeof(unsigned int);
}
//****************************************************************************************************************
//Fce pro pridani diry pro adresu navesti
//****************************************************************************************************************
void CAssembler::AddHoleForLabel()
{
	CheckCodeSize(sizeof(unsigned int));

	m_iCurrPositionInCode += sizeof(unsigned int);
}
//****************************************************************************************************************
//Funkce pro zpracovani der do kterych se maji ulozit adresy navesti
//****************************************************************************************************************
void CAssembler::processHoles()
{
	unsigned int labelscount = m_LabelsHoles.Size();		//zjisti pocet der
	unsigned int adress = 0;								//pomocna promenna pro adresu

	for (unsigned int i = 0; i < labelscount; i++)			//projdi cely seznam der
	{
		LABEL_ADRESSES* pLabel = m_LabelsHoles[i];			

		//tady se zeptame na adresu navesti podle unikatniho jmena navesti
		adress = (unsigned int)m_LabelsPositions.GetLabelAdress(pLabel->unique_labelname);
		//doplnime ziskanou adresu do priparvene diry
		*((unsigned int*)(&m_pByteCode[pLabel->label_offset])) = adress;
	}
}
//****************************************************************************************************************
//fce pro prevod instrukce na bytove kody
//****************************************************************************************************************
void CAssembler::processCode(char* token)
{
	//Prevod textove podoby instrukci na jjeich bytove reprezentace
	//Nektere instrukce, jako instrukce skoku maji parametr adresu kam se ma skocit
	//v tom pripade musime taky pridat diru pro ulozeni teto adresy, adresa bude doplnena pozdeji
	//protoze v tento okamzik nemusi byt znama, jelikoz adresa na kterou se ma skocit muze
	//byt az za mistem ktere se zrovna zpracovava
	//Instrukce pro presuny dat maji vetsinou parametr index, ktery ukazuje do tabulky konstant nebo na promennou
	//tento index jednoduse doplnime
	int iparam;
	char* strparam;
	//musime projit vsechny moznosti prislusejici konstantam v opcodes.h
	if (strcmp(token,instruction_nop) == 0)	
		AddCode(nopCI);
	else if (strcmp(token,instruction_mul) == 0)
		AddCode(mulCI);
	else if (strcmp(token,instruction_neg) == 0)
		AddCode(negCI);
	else if (strcmp(token,instruction_mod) == 0)
		AddCode(modCI);
	else if (strcmp(token,instruction_sub) == 0)
		AddCode(subCI);
	else if (strcmp(token,instruction_div) == 0)
		AddCode(divCI);
	else if (strcmp(token,instruction_add) == 0)
		AddCode(addCI);
	else if (strcmp(token,instruction_lgoto) == 0)
	{
		AddCode(lgotoCI);							//pridej kod goto
		strparam = getToken(m_inputFile);			//precti parametr
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);	//pridej do seznamu diru
		AddHoleForLabel();							//udelej v kodu misto pro doplneni adresy
	}
	else if (strcmp(token,instruction_ifeq) == 0)
	{
		AddCode(ifeqCI);
		strparam = getToken(m_inputFile);
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);
		AddHoleForLabel();
	}
	else if (strcmp(token,instruction_ifne) == 0)
	{
		AddCode(ifneCI);
		strparam = getToken(m_inputFile);
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);
		AddHoleForLabel();
	}
	else if (strcmp(token,instruction_if_cmpeq) == 0)
	{
		AddCode(if_cmpeqCI);
		strparam = getToken(m_inputFile);
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);
		AddHoleForLabel();
	}
	else if (strcmp(token,instruction_if_cmpgt) == 0)
	{
		AddCode(if_cmpgtCI);
		strparam = getToken(m_inputFile);
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);
		AddHoleForLabel();
	}
	else if (strcmp(token,instruction_if_cmplt) == 0)
	{
		AddCode(if_cmpltCI);
		strparam = getToken(m_inputFile);
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);
		AddHoleForLabel();
	}
	else if (strcmp(token,instruction_if_cmpge) == 0)
	{
		AddCode(if_cmpgeCI);
		strparam = getToken(m_inputFile);
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);
		AddHoleForLabel();
	}
	else if (strcmp(token,instruction_if_cmple) == 0)
	{
		AddCode(if_cmpleCI);
		strparam = getToken(m_inputFile);
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);
		AddHoleForLabel();
	}
	else if (strcmp(token,instruction_if_cmpne) == 0)
	{
		AddCode(if_cmpneCI);
		strparam = getToken(m_inputFile);
		m_LabelsHoles.AddLabel(stringConcat(m_CurrFuncName,"_",strparam,NULL),m_iCurrPositionInCode);
		AddHoleForLabel();
	}
	else if (strcmp(token,instruction_nreturn) == 0)
		AddCode(nreturnCI);
	else if (strcmp(token,instruction_vreturn) == 0)
		AddCode(vreturnCI);
	else if (strcmp(token,instruction_load) == 0)
	{
		AddCode(loadCI);
		strparam = getToken(m_inputFile);
		iparam = atoi(strparam);
		AddIndex16((unsigned short)iparam);
	}
	else if (strcmp(token,instruction_store) == 0)
	{
		AddCode(storeCI);
		strparam = getToken(m_inputFile);
		iparam = atoi(strparam);
		AddIndex16((unsigned short)iparam);
	}
	else if (strcmp(token,instruction_ldc_int) == 0)
	{
		AddCode(ldc_intCI);
		strparam = getToken(m_inputFile);		//precteni parametru
		iparam = atoi(strparam);	//konstanta ktera se ma nacist
		AddIndex32(m_Constants.GetIntNumber(iparam));	//index do tabulky konstant
	}
	else if (strcmp(token,instruction_ldc_string) == 0)
	{
		AddCode(ldc_stringCI);
		strparam = getStringToken(m_inputFile);
		AddIndex32(m_Constants.GetStringNumber(strparam));	//index do tabulky konstant
	}
	else if (strcmp(token,instruction_ldc_double) == 0)
	{
		AddCode(ldc_doubleCI);
		token = getToken(m_inputFile);
		double fparam = atof(token);
		AddIndex32(m_Constants.GetDoubleNumber(fparam));
	}
	else if (strcmp(token,instruction_dup) == 0)
		AddCode(dupCI);
	else if (strcmp(token,instruction_pop) == 0)
		AddCode(popCI);
	else if (strcmp(token,instruction_lcall) == 0)
	{
		AddCode(lcallCI);
		strparam = getToken(m_inputFile);
		AddIndex32(m_Constants.GetStringNumber(strparam));
	}
	else if (strcmp(token,instruction_ecall) == 0)
	{
		AddCode(ecallCI);
		strparam = getToken(m_inputFile);
		AddIndex32(m_Constants.GetStringNumber(strparam));
		iparam = atoi(getToken(m_inputFile));
		AddIndex16((unsigned short)iparam);
	}
	else if (strcmp(token,instruction_shl) == 0)
		AddCode(shlCI);
	else if (strcmp(token,instruction_shr) == 0)
		AddCode(shrCI);
	else if (strcmp(token,instruction_inc) == 0)
		AddCode(incCI);
	else if (strcmp(token,instruction_dec) == 0)
		AddCode(decCI);
	else if (strcmp(token,instruction_or) == 0)
		AddCode(orCI);
	else if (strcmp(token,instruction_and) == 0)
		AddCode(andCI);
	else
		theLog.ReportError(line,"FATAL ERROR: Unrecognized '%s' token in source file!<br>",token);
}
//****************************************************************************************************************
//fce pro vypsani tabulky konstant do souboru
//****************************************************************************************************************
void CAssembler::WriteConstants()
{
	CONSTANT* pConstant = NULL;
	char	id = CONSTANTS_ID;
	unsigned int numconst = m_Constants.Size();

	fwrite(&id,sizeof(char),1,m_binFile);
	fwrite(&numconst,sizeof(unsigned int),1,m_binFile);
	
	for (unsigned int i = 0; i < numconst; i++)								//jde jen o prochazeni pole konstant, zjistovani o jaky typ
	{																		//konstanty jde a zapsani tzpu konstanty a prislusne hodnoty do souboru
		pConstant = m_Constants[i];
		fwrite(&(pConstant->type),sizeof(char),1,m_binFile);
		switch (pConstant->type)
		{
		case INT_CONST:
			fwrite(&(pConstant->val.intval),sizeof(int),1,m_binFile);
			break;
		case ZSTRING_CONST:
			WriteZString(pConstant->val.str,m_binFile);
			break;
		case DOUBLE_CONST:
			fwrite(&(pConstant->val.doubleval),sizeof(double),1,m_binFile);
			break;
		}
	}
}

//****************************************************************************************************************
//fce pro vypsani kodoveho segmentu do souboru
//****************************************************************************************************************
void CAssembler::WriteCodeSegment()
{
	int iRet = 1;
	unsigned char code = CODE_ID;

	fwrite(&code,sizeof(char),1,m_binFile);	//zapsani informaci o tom ze se jedna o kodovy segment

    iRet = fwrite(&m_iCurrPositionInCode,sizeof(unsigned int),1,m_binFile);			//zapsani velikosti kodoveho segmentu

	iRet = fwrite(m_pByteCode,m_iCurrPositionInCode * sizeof(unsigned char),1,m_binFile);	//zapsani vlastniho kodoveho segmentu
}

//****************************************************************************************************************
//fce pro prevedeni souboru z assembly do binarni formy
//****************************************************************************************************************
void CAssembler::Process(char* pFile)
{
	//nejdriv sestavime jmeno vystupniho souboru
	char* outputFile = new char[strlen(pFile) + 1];
	strcpy(outputFile,pFile);
	strcpy(outputFile + strlen(outputFile) - strlen(BINARY_EXT),BINARY_EXT);

	//otevreni vstupniho assembly souboru
	m_inputFile = fopen(pFile,"r");
	if (!m_inputFile)
	{
		theLog.TraceC("ERROR: Cannot open input file '%s' for reading!\n",pFile);	//hlaska o chybe
		theLog.TraceF("ERROR: Cannot open input file '%s' for reading!<br>",pFile);	//hlaska o chybe
		return;
	}

	//otevreni vystupniho binarniho souboru
	m_binFile = fopen(outputFile,"wb");
	if (!m_binFile)
	{
		theLog.TraceC("ERROR: Could'nt open output file '%s' for writing!\n",outputFile);	//hlaska o chybe
		theLog.TraceF("ERROR: Could'nt open output file '%s' for writing!<br>",outputFile);	//hlaska o chybe
		return;
	}

	//zapsani podpisu souboru, jen takova prkotina pro prvni konstrolu zda jde o soubor skriptu
	char* filesignature = LANGUAGE_SIGNATURE;
	fwrite(filesignature,3*sizeof(char),1,m_binFile);

	char* token = getToken(m_inputFile);	//precteni symbolu ze vstupniho souboru

	while (token)	
	{
		//v teto chvili se ocekava jen symbol .function, ze ktereho by se mely skripty skladat, nebo symbol .import
		//jiny symbol je v tuto chvili neplatny
		if (strcmp(token,token_function) == 0)
			processFUNCTION();
		else if (strcmp(token,token_import) == 0)
			processIMPORT();
		else
			theLog.ReportError(line,"Illegal top-level token '%s' in source file: '%s'\n",token,pFile);

		token = getToken(m_inputFile);
	}
	
	//doplneni adres navesti k instrukcim skoku
	processHoles();
	//vypis tabulky konstant
	WriteConstants();
	//vypis kodoveho segmentu
	WriteCodeSegment();
	//zavreni souboru
	fclose(m_inputFile);
	fclose(m_binFile);
}
//****************************************************************************************************************
//Funkce pro zpracovani funkce
//****************************************************************************************************************
void CAssembler::processFUNCTION()
{
	FUNCTION_HEADER* fh = new FUNCTION_HEADER;
	char			id;
	char*			token = NULL;
	char*			tmp = NULL;
	int				iparam = 0;

	//nulovani struktury
	memset(fh,0,sizeof(FUNCTION_HEADER));

	fh->startCode = m_iCurrPositionInCode;		//ulozeni adresy kde funkce zacina

	//musime nacist jmeno funkce
    token = getToken(m_inputFile);
	tmp = token;		//docasne ulozeni ukazatele

	//pocitame dokud nenajdeme (, kde jsou parametry, treba fce min(2), tj min, dva parametry
	for (tmp; *tmp != '(';tmp++);

	//ted mame v end pozici kde konci jmeno a je zavorka(
	//nacteni poctu argumentu
	tmp++;
	char numArguments[128];
	int index = 0;
	while(*tmp != ')')
	{
		numArguments[index++] = *tmp;
		tmp++;
	}
	numArguments[index] = 0;
	fh->inputs = atoi(numArguments);

	//ted se rozhodne kolik ma vystupnich parametru, jeden nebo zadny
	if (*(++tmp) == '1')
		fh->ouputs = 1;
	else
		fh->ouputs = 0;

	fh->name = token;	//ulozeni jmena fce
	m_CurrFuncName = token;

	do
	{
		token = getToken(m_inputFile);		//precteni symbolu ze souboru

		if (IsLabel(token))					//pokud nacteny symbol je navesti
		{
			token[strlen(token) - 1] = 0;	//zrus dvojtecku na konci navesti
			m_LabelsPositions.AddLabel(stringConcat(m_CurrFuncName,"_",token,NULL),m_iCurrPositionInCode);	//uloz adresu navesti do seznamu
		}	
		else if (strcmp(token,token_locals_limit) == 0)		//symbol udava lokalni limit funkce
		{
			token = getToken(m_inputFile);
			iparam = atoi(token);
			fh->localsLimit = iparam;
		}
		else if (strcmp(token,token_stack_limit) == 0)		//symbol udava limit zasobniku funkce
		{
			token = getToken(m_inputFile);
			iparam = atoi(token);
			fh->stackLimit = iparam;
		}
		else if (strcmp(token,token_end_function) == 0)
		{
			; //nic nedelej
		}
		else
			processCode(token);		//assembly tela funkce
	}
	while (token && (strcmp(token,token_end_function) != 0));

	fh->endCode = m_iCurrPositionInCode;		//ulozeni adresy kde funkce konci
	//ulozeni ziskanych informaci do souboru
	id = FUNCTION_ID;	//oznaceni ze se jedna o blok definice funkce
	fwrite(&id,sizeof(char),1,m_binFile);
	WriteZString(fh->name,m_binFile);
	fwrite(&(fh->inputs),sizeof(unsigned short),1,m_binFile);
	fwrite(&(fh->ouputs),sizeof(unsigned short),1,m_binFile);
	fwrite(&(fh->localsLimit),sizeof(unsigned short),1,m_binFile);
	fwrite(&(fh->stackLimit),sizeof(unsigned short),1,m_binFile);
	fwrite(&(fh->startCode),sizeof(unsigned int),1,m_binFile);
	fwrite(&(fh->endCode),sizeof(unsigned int),1,m_binFile);

	theLog.TraceF("Function %s, inputs: %i, outputs: %i, code size: %i<br>",fh->name,fh->inputs,fh->ouputs,fh->endCode - fh->startCode);
}
//****************************************************************************************************************
//Funkce pro zpracovani importovane funkce
//****************************************************************************************************************
void CAssembler::processIMPORT()
{
	IMPORT* pImport = new IMPORT;

	pImport->library = getStringToken(m_inputFile);	//preteni knihovny v ktere lze funkci najit

	char* type = getToken(m_inputFile);			//zjisteni a ulozeni typu ktery funkce vraci
	if (strcmp(type,token_type_int) == 0)
		pImport->type = INT_CONST;
	else if (strcmp(type,token_type_string) == 0)
		pImport->type = ZSTRING_CONST;
	else if (strcmp(type,token_type_double) == 0)
		pImport->type = DOUBLE_CONST;
	else
		pImport->type = VOID_TYPE;
	
	char* signature = getToken(m_inputFile);	//precteni podpisu funkce
	char* name = signature;	
	int   end = 0;
	char  inputs[128];
	int	  index = 0;

	//ulozeni kopie podpisu pomoci funkce pro slozeni retezcu
	pImport->signature = stringConcat(signature,NULL);		
	//****ziskani jmena funkce odstranenim casti se zavorkami z podpisu*****//
	for (signature;*signature != '('; signature++,end++);

	name[end] = 0;

	pImport->name = name;
	//***********************************************************************//
	

	//ted zapis do souboru
	char code = IMPORT_ID;

	fwrite(&code,sizeof(char),1,m_binFile);

	WriteZString(pImport->name,m_binFile);
	WriteZString(pImport->signature,m_binFile);
	WriteZString(pImport->library,m_binFile);
	//fwrite(&(pImport->inputs),sizeof(unsigned short),1,m_binFile);
	fwrite(&(pImport->type),sizeof(char),1,m_binFile);
}